//
//  BPOfflineManager.h
//  BIViews
//
//  Created by Brent Gulanowski on 10-04-01.
//  Copyright 2010 Marketcircle Inc. All rights reserved.
//

#import <Cocoa/Cocoa.h>

#import <BPAdmin/BPSyncHelper.h>
#import <MCClientServer/MCClientServerDefines.h>


#define BP_BUILD_FOR_MAC_APP_STORE true


typedef enum {
    BPCredentialVerificationAfterSync = 0,
    BPCredentialVerificationDuringAppLaunch = 1
} BPCredentialVerificationMode;

@class BPDaemonKeeper;

@interface BPOfflineManager : NSObject {

	id progressDelegate;
	NSString *notificationIdentifier;
	BPDaemonKeeper *daemonKeeper;
	
#if BP_BUILD_FOR_MAC_APP_STORE
	id<BPSyncHelper> syncHelper; // DO proxy
#endif
	
	NSTimer *syncAbortTimer;
	
	BOOL creating;
	BOOL manualSync;
	BOOL showDeterminateProgress;
	BOOL helperEnabled;
    BOOL isSynchronizing;
	
	NSDate *lastSyncDateWithChanges;
	NSNumber *lastSyncWithChangesRowsProcessed;
    NSDate *lastSyncAttemptDate;
    
    BPCredentialVerificationMode verificationMode;
}

@property (assign) id progressDelegate;
@property (retain) NSDate *lastSyncDateWithChanges;
@property (retain) NSNumber *lastSyncWithChangesRowsProcessed;
@property (retain) NSDate *lastSyncAttemptDate;
@property (assign) BOOL isSynchronizing;

+ (BPOfflineManager *)sharedManager;

- (BOOL)createOfflineWithUser:(NSString *)appUser
						 pass:(NSString *)appPass
			  accessAddresses:(NSArray *)accessAddresses
				   accessPort:(NSNumber *)accessPort
				  pgAddresses:(NSArray *)addresses
					   pgPort:(NSNumber *)dbPort
				cloudUsername:(NSString *)cloudUsername
				   dataDBName:(NSString *)dataDBName
					   dbUser:(NSString *)dbUser
					   dbPass:(NSString *)dbPass
					   dbUUID:(NSString *)dbUUID;

- (BOOL)createOfflineWithUser:(NSString *)appUser
						 pass:(NSString *)appPass
			  accessAddresses:(NSArray *)accessAddresses
				   accessPort:(NSNumber *)accessPort
				  pgAddresses:(NSArray *)addresses
					   pgPort:(NSNumber *)dbPort
				cloudUsername:(NSString *)cloudUsername
				   dataDBName:(NSString *)dataDBName
					   dbUser:(NSString *)dbUser
					   dbPass:(NSString *)dbPass
					   dbUUID:(NSString *)dbUUID
					 dbSchema:(NSString *)dbSchema;


- (void)showVerifyCredentialsWindowForUsername:(NSString *)username verificationMode:(BPCredentialVerificationMode)mode; 

/*!
 Check the connection, restart it if necessary and then syncs
 */
- (void)startSync;

/*!
 Just sends the sync message to sync helper - doesn't check the connection
 */
- (void)syncWithoutChecking;

- (BOOL)isSyncingManually;

- (BOOL)enableSyncHelperLoginItem:(BOOL)flag;

/*!
 Returns MCSAuthenticationNoError if all is good. The main window should not open otherwise. If it is credential related, someone can listen to verified creditials notification and then open the main window
 */
- (MCSAuthenticationErrorCode)checkOfflineStatus:(NSNumber *)errorCode;

- (void)observeSyncNotifications;

/*!
 Reads from the last LastSyncAttemptInfo.plist (that sync helper writes if there are changes) and updates the instance vars. Called at init and after a sync notif.
 */
- (void)updateLastSyncAttemptInfo;

/*!
 Reads from the LastSyncInfo plist (that sync helper writes if there are changes) and updates the instance vars. Called at init and after a successful sync notif.
 */
- (void)updateLastSyncWithChangesInfo;

/*!
 Returns a string that denotes the last successful sync with changes - put the change count in brackets. Caller must handle a nil case. We only make this string if we have the info.
 */
- (NSString *)lastSyncInfoStringWithChangesForMenu;

/*!
 Checks to see if the is a IsSynchronizing.txt file to determine if the sync helper was already synching prior to Billings Pro starting
 */
- (void)updateSyncStatusIfCurrentlySynchronizing;

/*!
 Unregisters the sync helper, stops it and then registers it. If it is unable to connect over DO, it issues a killall.
 */
- (void)stopSyncHelper;

/*!
 Unregisters the sync helper and asks it to stop - does not try to connect
 */
- (void)stopSyncHelperWithoutForcingConnect;


/*!
 Registers the sync helper
 */
- (void)startSyncHelper;


/*!
 Calls stopSyncHelper and then startSyncHelper.
 */
- (void)restartSyncHelper;

/*!
 Unregisters the sync helper and the starts it again. In this case we do not try to connect and stop. This for cases where sync helper has quit unexpectedly.
 */
- (void)restartSyncHelperWithoutForcingStop;

/*!
 When the MAS updates BP and the user has chosen to auto-sync, it doesn't restart the sync tool. Sometimes, this causes the sync helper to hang indefinitely. With this call, we check the last version bundle version and if it is different, we stop the process. Launchd should restart it immediatly and thus we get a new process.
 */
- (void)restartSyncHelperIfNewVersion;


/*!
 Uses NSRunningApplication to check if the sync helper is running and if so, checks if the connection is known to be valid. These are lightweight checks.
 */
- (BOOL)isSyncHelperRunningQuickCheck;


@end


@interface NSObject (BPOfflineManagerProgressDelegate)

- (NSWindow *)progressWindow;
- (NSProgressIndicator *)progressIndicator;
- (BOOL)showDeterminateProgress;

@end

